using Microsoft.Web.Services;
using Microsoft.Web.Services.Security;
using Microsoft.Web.Services.Security.X509;

using System;
using System.Text;
using System.Web.Services.Protocols;

namespace WSSSignature {
	/// <summary>
	/// La classe Signer rcupre le jeton de scurit associ au certificat
	/// X509 choisi par l'utilisateur et s'en sert pour signer le message SOAP
	/// d'invocation du service Web de cotation d'actions WSE (.NET) ou WSTK
	/// (Java).
	/// </summary>
	class Signer {
		string symbol;
		string subjectName;
		string implementation;

		/// <summary>
		/// La classe reconnat 2 (minimum) ou 3 paramtres :
		/// 1. le symbole de la valeur boursire (obligatoire);
		/// 2. le nom de l'objet du certificat X509 (obligatoire);
		/// 3. le service de cotation  invoquer (facultatif : "WSE"
		/// correspond au service .NET, "WSTK" correspond au service
		/// Java et toute autre valeur correspond  une invocation 
		/// des 2 services).
		/// </summary>
		Signer(string[] args) {
			symbol = args[0];
			subjectName = args[1];
			implementation = null;
			if (args.Length == 3) {
				implementation = args[2];
			}
		}

		/// <summary>
		/// Point d'entre de l'application.
		/// </summary>
		[STAThread]
		static void Main(string[] args) {
			Signer signer = null;
			try {
				signer = new Signer(args);
				if (signer.implementation == "WSE") {
					signer.SignWSE();
					return;
				}
				if (signer.implementation == "WSTK") {
					signer.SignWSTK();
					return;
				}
				signer.SignWSE();
				signer.SignWSTK();
			}
			catch (Exception e) {
				Console.WriteLine(e);
				Console.WriteLine("\nL'un des paramtres est incorrect ou absent.");
				return;
			}
		}
        
		/// <summary>
		/// Signature et invocation du service Web .NET.
		/// </summary>
		private void SignWSE() {
			Console.WriteLine("\nSignature et invocation du service Web .NET...");

			// Cration d'une instance du proxy-service WSE.
			ProxyWSE proxyWSE = new ProxyWSE();
			SoapContext requestContext = proxyWSE.RequestSoapContext;

			// Rcupration du jeton de scurit X509.
			X509SecurityToken token = GetSecurityToken(subjectName);
			if (token == null) {
				throw new ApplicationException("Pas de cl de signature trouve.");
			}

			// Ajout de la signature  la requte SOAP.
			requestContext.Security.Tokens.Add(token);
			requestContext.Security.Elements.Add(new Signature(token));
			requestContext.Path.MustUnderstand = false;

			// Invocation du service Web .NET.
			Console.WriteLine("\nInvocation de : {0}", proxyWSE.Url);
			float value = proxyWSE.GetQuote(symbol);

			// Affichage du rsultat de l'invocation.
			string message = string.Format("{0} : {1}", symbol, value);
			Console.WriteLine("Valeur de l'action {0}", message);
		}

		/// <summary>
		/// Signature et invocation du service Web Java.
		/// </summary>
		private void SignWSTK() {
			Console.WriteLine("\nSignature et invocation du service Web Java...");

			// Cration d'une instance du proxy-service WSTK.
			ProxyWSTK proxyWSTK = new ProxyWSTK();
			SoapContext requestContext = proxyWSTK.RequestSoapContext;

			// Rcupration du jeton de scurit X509.
			X509SecurityToken token = GetSecurityToken(subjectName);
			if (token == null) {
				throw new ApplicationException("Pas de cl de signature trouve.");
			}

			// Ajout de la signature  la requte SOAP.
			requestContext.Security.Tokens.Add(token);
			requestContext.Security.Elements.Add(new Signature(token));
			requestContext.Path.MustUnderstand = false;

			// Invocation du service Web Java.
			Console.WriteLine("\nInvocation de : {0}", proxyWSTK.Url);
			float value = proxyWSTK.getQuote(symbol);

			// Affichage du rsultat de l'invocation.
			string message = string.Format("{0} : {1}", symbol, value);
			Console.WriteLine("Valeur de l'action {0}", message);            
		}

		/// <summary>
		/// Rcupration du jeton de scurit X509  utiliser pour
		/// signer les messages SOAP.
		/// </summary>
		/// <returns>Le jeton de scurit X509.</returns>
		X509SecurityToken GetSecurityToken(string subjectName) {            
			X509SecurityToken securityToken;  

			// Ouverture du magasin de certificats de l'utilisateur.
			X509CertificateStore store = X509CertificateStore.CurrentUserStore(X509CertificateStore.MyStore);
			store.OpenRead();

			try {
				// Recherche du certificat slectionn par l'utilisateur.
				X509Certificate certificate;
				X509CertificateCollection matchingCerts = store.FindCertificateBySubjectString(subjectName);
				if (matchingCerts.Count == 0) {
					throw new ApplicationException("Aucun certificat trouv, associ au nom de l'objet pass en paramtre.");
				}
				else {
					certificate = matchingCerts[0];
				}

				// Le certificat est-il adquat pour la signature numrique ?
				if (!certificate.SupportsDigitalSignature || certificate.Key == null ) {
					throw new ApplicationException("Le certificat choisi ne supporte pas les signatures numriques ou ne dispose pas d'une cl prive.");
				}
				else {
					byte[] keyId = certificate.GetKeyIdentifier();
					Console.WriteLine("\nObjet du certificat choisi : {0}", certificate.GetName());
					Console.WriteLine("Identificateur de la cl   : {0}", Convert.ToBase64String(keyId));
					Console.WriteLine("Cl publique               : {0}", certificate.GetPublicKeyString());
					securityToken = new X509SecurityToken(certificate);
				}
			} 
			finally {
				if (store != null) { 
					store.Close(); 
				}
			}            
			return securityToken;            
		}
	}
}
